# Voidburn Enterprise Install (EKS) - Deterministic Compute Spend Governance

Voidburn is a deterministic governance layer for compute spend.
It enforces budgets with scoped controls, proof-gated termination, and audit-grade evidence artifacts.

This installer is designed for enterprise environments:
- least-privilege by default
- supply chain verification (cosign)
- SBOM evidence
- deterministic strict checkpoint validation
- evidence bundle output for security/procurement

## What this install does (high level)
1. Validates target scope:
   - nodegroup: ASG-backed + ON_DEMAND (required for guaranteed resume on ASG-backed capacity)
   - karpenter: NodePool/Provisioner exists (Karpenter must be installed)
2. Ensures a scaler exists (Karpenter OR Cluster Autoscaler)
3. Installs Sentinel via one CloudFormation stack
4. Deploys (or references) a strict checkpoint workload endpoint (POST /checkpoint)
5. Verifies strict mode end-to-end (receiver -> workload -> marker)
6. Runs IAM simulation proof that Sentinel is tag-scoped (tagged ALLOW; untagged DENY)
7. Produces an evidence bundle: `voidburn-evidence-<timestamp>.tar.gz`

## Enterprise quickstart (successful install)
### 0) Pin a vendor release (manifest)
Pin to a specific release for change-controlled installs. Use `latest` only for non-production testing.

```bash
MANIFEST_URL="https://voidburn.com/releases/v1.2.10/RELEASE_MANIFEST.json"
# For non-production testing only:
# MANIFEST_URL="https://voidburn.com/releases/latest/RELEASE_MANIFEST.json"
```

### 1) Create `env.customer`
```bash
cp examples/env.example env.customer
$EDITOR env.customer
```

### 2) Pin signed image + cosign contract into `env.customer`
This appends exported variables (so the installer can see them):

```bash
curl -fsS "$MANIFEST_URL" | jq -r '
  "export COSIGN_ISSUER=" + (.cosign.issuer|@sh),
  "export COSIGN_IDENTITY_REGEX=" + (.cosign.identityRegex|@sh),
  "export WORKLOAD_IMAGE=" + (.images.workloadCheckpoint.ref|@sh)
' >> env.customer
```

Optional: verify supply chain before installing:

```bash
source env.customer
cosign verify \
  --certificate-oidc-issuer "$COSIGN_ISSUER" \
  --certificate-identity-regexp "$COSIGN_IDENTITY_REGEX" \
  "$WORKLOAD_IMAGE"
cosign download attestation "$WORKLOAD_IMAGE" | head
```

### 3) Install
```bash
source env.customer
./voidburnctl-install-eks.sh
```

### 4) Prove enforcement (in the app)
After the first heartbeat is active:
- Create a CLUSTER budget with a low limit to force a breach.
- Watch Operations/Overview for: checkpoint confirmed -> snapshot -> termination evidence.
- The cluster becomes FROZEN. Resume is manual (click Resume when you want capacity back).

## Requirements
### On the operator machine
- aws cli v2
- kubectl
- eksctl
- jq
- cosign
- openssl

### Cluster requirements
- EKS cluster reachable from operator network
- At least one ASG-backed nodegroup
- Target scope:
  - If targeting a nodegroup: it must be ON_DEMAND for "guaranteed resume"
  - If targeting a Karpenter NodePool/Provisioner: Karpenter must be installed and healthy
- Protected capacity (required): Sentinel is pinned to `voidburn.com/protected=true`. In Karpenter target mode, protected capacity must be outside the governed NodePool so Sentinel stays running while targets are frozen.
- If using strict + efs: EFS CSI + RWX PVC must exist (stack typically creates this)

## Supply chain verification (cosign)
You must use ONE verification mode:

### Option A - Keyless verification (recommended)
Set:
- COSIGN_ISSUER
- COSIGN_IDENTITY_REGEX

### Option B - Public key verification
Set:
- COSIGN_PUBKEY=/path/to/voidburn-cosign.pub

If verification fails, install fails.

## Workload checkpoint image (enterprise)
You must provide ONE of:
- WORKLOAD_IMAGE: an internal, approved image reference (digest recommended)
- KANIKO_PUSH_IMAGE: destination image for in-cluster build

If you use WORKLOAD_IMAGE and the registry needs auth:
- Create a pull secret in vb-workloads and set IMAGE_PULL_SECRET

## Run
1. Copy `examples/env.example` to `env.customer`
2. Fill values (except API key; that is prompted securely)
3. Pin release values from the vendor manifest (recommended)
4. Run:
   `source env.customer && ./voidburnctl-install-eks.sh`

## Output
- Evidence bundle: `voidburn-evidence-<timestamp>.tar.gz`
- Includes: IAM simulation proof JSON, cosign verify JSON, SBOM (if present), strict smoke result, logs, ASG snapshots, node label snapshot

## Uninstall
`source env.customer && ./voidburnctl-uninstall-eks.sh`

Security note: This installer never asks for inbound access. The agent is outbound-only.
